<#
    .SYNOPSIS
    Configures a computer to act as a Kape collection server. 

    .DESCRIPTION
    Configures a computer to act as a Kape collection server. It configures the network settings, creates a temporary user account, shares the Kape executable folder and the Kape results folder, and opens port 445 for inbound connections from the target computers. A Settings.psd1 configuration file is used to ensure that the same settings are used for the server configuration and collection scripts.

    .PARAMETER SettingsFile
    The path to the settings.psd1 file. If not specified, the script will look for the file in the same directory as the script.
    
    .EXAMPLE
    PS> Prepare-AutoKapeServer.ps1
    This looks for Settings.psd1 in the same directory as the script and configures the computer it runs on based on the arguments specified in Settings.psd1. 
    
    .EXAMPLE
    PS> Prepare-AutoKapeServer.ps1 -SettingsFile C:\Users\user\Desktop\Settings.psd1
    This configures the system with the arguments specified in C:\Users\user\Desktop\Settings.psd1.

    .NOTES
    Written by Steve Anson. This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version. This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. The GNU General Public License can be found at <https://www.gnu.org/licenses/>.
    
    .LINK
    You can download Kape from here: https://www.kroll.com/en/services/cyber-risk/incident-response-litigation-support/kroll-artifact-parser-extractor-kape
#>
param (
    [string]$SettingsFile = (Join-Path -Path $PWD -ChildPath 'Settings.psd1'),
    [string[]]$TargetComputers,
    [string]$ServerUserName,
    [SecureString]$ServerUserPassword,
    [string]$ServerIPAddress,
    [string]$ServerSubnetMask,
    [string]$DNSServer1,
    [string]$DNSServer2,
    [string]$DefaultGateway,
    [string]$KapeExePath,
    [string]$KapeExeSharedFolderName,
    [string]$DestinationPath,
    [string]$DestinationSharedFolderName,
    [string]$ServerInterfaceName,
    [string]$ServerTranscriptPath,
    [string]$ServerTranscriptName,
    [string]$RunTranscriptPath,
    [string]$RunTranscriptName,
    [string]$KapeArguments    
)

# Check if the current PowerShell session is running as administrator

try {
    $currentPrincipal = [Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()
    if (-NOT $currentPrincipal.IsInRole([Security.Principal.WindowsBuiltInRole] "Administrator")) {
        throw 
    }
}
catch {
    Write-Host -ForegroundColor Red "ERROR: This script must be run as Administrator."
    Write-Host -ForegroundColor Red "No changes have been made to this system."
    exit 1
}

# Import the .psd1 file
if (-not (Test-Path -Path $SettingsFile)) {
    Write-Host -ForegroundColor Red "ERROR: $SettingsFile file not found."
    Write-Host -ForegroundColor Red "Make sure that the file exists or specify a different location with the -SettingsFile parameter."
    Write-Host -ForegroundColor Red "No changes have been made to this system."
    Exit 1
}

$variables = Import-PowerShellDataFile -Path 'settings.psd1'

# Assign values from the imported file to the parameters
$TargetComputers = $variables.TargetComputers
$ServerUserName = $variables.ServerUserName
if ($variables.ServerUserPassword) {
    $ServerUserPassword = ConvertTo-SecureString -String $variables.ServerUserPassword -AsPlainText -Force
}
$ServerIPAddress = $variables.ServerIPAddress
$ServerSubnetMask = $variables.ServerSubnetMask
$DNSServer1 = $variables.DNSServer1
$DNSServer2 = $variables.DNSServer2
$DefaultGateway = $variables.DefaultGateway
$KapeExePath = $variables.KapeExePath
$KapeExeSharedFolderName = $variables.KapeExeSharedFolderName
$KapeExeShare = "\\$ServerIPAddress\$KapeExeSharedFolderName"
$DestinationPath = $variables.DestinationPath
$DestinationSharedFolderName = $variables.DestinationSharedFolderName
$DestinationShare = "\\$ServerIPAddress\$DestinationSharedFolderName"
$ServerInterfaceName = $variables.InterfaceName
$ServerTranscriptPath = $variables.ServerTranscriptPath
$ServerTranscriptName = $variables.ServerTranscriptName
$RunTranscriptPath = $variables.RunTranscriptPath
$RunTranscriptName = $variables.RunTranscriptName
$KapeArguments = $variables.KapeArguments

# Get ServerUserPassword if not provided by the settings file
if (-NOT $ServerUserPassword) {
    Write-Host "Please enter the password to be used for the $ServerUserName account."
    $ServerUserPassword = Read-Host -AsSecureString
}

#Start Transcript
if (-NOT (Test-Path -Path $ServerTranscriptPath)) {
    Write-Host "Creating $ServerTranscriptPath..."
    New-Item -ItemType Directory -Path $ServerTranscriptPath -Force
}

$timestamp = Get-Date -Format "yyyyMMdd_HHmmss"
$DatedTranscriptName = "$ServerTranscriptName-$timestamp"
$ServerTranscript = Join-Path -Path $ServerTranscriptPath -ChildPath $DatedTranscriptName

Start-Transcript -Path $ServerTranscript
Write-Host "Script transcript starting at $(Get-Date -Format 'yyyy-MMM-dd HH:mm:ss')."

# Verify local environment
if (-NOT (Test-Path -Path $KapeExePath)) {
    Write-Host -ForegroundColor Red "ERROR: $KapeExePath does not exist. Please create it and place the Kape files inside."
    Write-Host -ForegroundColor Red "No changes have been made to this system."
    exit 1
}

if (-NOT (Test-Path -Path $DestinationPath)) {
    Write-Host -ForegroundColor Red "ERROR: $DestinationPath does not exist. Please create it and make sure it is empty."
    Write-Host -ForegroundColor Red "No changes have been made to this system."
    exit 1
}

if ((Get-ChildItem -Path $DestinationPath)) {
    Write-Host -ForegroundColor Red "ERROR: $DestinationPath is not empty. Please make sure it is empty."
    Write-Host -ForegroundColor Red "No changes have been made to this system."
    exit 1
}

#Check if $ServerUserName exists
if (Get-LocalUser -Name $ServerUserName -ErrorAction SilentlyContinue) {
    Write-Host -ForegroundColor Red "ERROR: $ServerUserName already exists. This script is intended to create a new user account."
    Write-Host -ForegroundColor Red "Since the account will be inherently insecure, the intent is to create the account only for the duration of the collection process."
    Write-Host -ForegroundColor Red "Please configure a different ServerUserName in Settings.psd1 and try again."
    Write-Host -ForegroundColor Red "No changes have been made to this system."
    exit 1
}

# Confirm that TargetComputers are resolvable
$ResolutionError = $false
foreach ($computer in $TargetComputers) {
    try {
        $ComputerIP = Resolve-DnsName -Name $computer -Type A -ErrorAction Stop | Select-Object -ExpandProperty IPAddress
    }
    catch {
        Write-Host -ForegroundColor Red "ERROR: An error occurred while resolving the IP address for $computer."
        Write-Host -ForegroundColor Red "$_"
        $ResolutionError = $true
    }
    if ($ResolutionError) {
        Write-Host -ForegroundColor Red "Please make sure that all Target Computers are can be resolved by DNS on $DNSServer1 or $DNSServer2 and try again."
        Write-Host -ForegroundColor Red "No changes have been made to this system."
        exit 1
    }
}

# Confirm configuration details

# Get a list of network adapter names
$adapterNames = (Get-NetAdapter).Name

Write-Host "Now we will confirm the settings that you want to use for this computer."
Write-Host "Here are the network adapter names on this computer."
Get-NetAdapter | Format-Table
Write-Host "So the valide network adapter names are:" 
$adapterNames | ForEach-Object { Write-Host $_ }

$inputvalue = Read-Host -Prompt "The interface in Settings.psd1 is $ServerInterfaceName.  Hit enter to accept or type a different interface name"
if ([string]::IsNullOrWhiteSpace($inputvalue))
{}
else {
$ServerInterfaceName = $inputvalue
}



# Keep prompting the user for a valid adapter name until valid
while ($adapterNames -notcontains $ServerInterfaceName) {
    $ServerInterfaceName = Read-Host -Prompt "The interface name $ServerInterfaceName is not valid. Please enter a valid network adapter name"
}

Write-Host "Now we will confirm the settings that you want to use for this computer."
Write-Host "TargetComputers: $TargetComputers"
Write-Host "ServerUserName: $ServerUserName"
Write-Host "ServerUserPassword: $($variables.ServerUserPassword)"
Write-Host "ServerIPAddress: $ServerIPAddress"
Write-Host "ServerSubnetMask: $ServerSubnetMask"
Write-Host "DNSServer1: $DNSServer1"
Write-Host "DNSServer2: $DNSServer2"
Write-Host "KapeExePath: $KapeExePath"
Write-Host "KapeExeShare: $KapeExeShare"
Write-Host "DestinationPath: $DestinationPath"
Write-Host "DestinationShare: $DestinationShare"
Write-Host "InterfaceName: $ServerInterfaceName"

#confirm that user wants to continue
Write-Host "Please confirm that these settings are correct.  If they are not, exit this script and edit the Settings.psd1 file."
while ($true) {
    $response = Read-Host "Do you want to continue? (Y/N)"
    if ($response -imatch "^(y|yes)$") {
        break
    }
    elseif ($response -imatch "^(N|No)$") {
        Write-Host "Exiting..."
        exit
    }
    else {
        Write-Host "Invalid response. Please enter Y or N."
    }
}

# Record the original network settings
Write-Host "Just to be safe, here are your network settings before this script makes any changes:"
ipconfig /all
Write-Host ""

# Make the designated changes

# Disable DHCP 
Write-Host "Now we will start making changes..."
Write-Host "Disabling DHCP."

try {
    Get-NetAdapter -Name $ServerInterfaceName | Set-NetIPInterface -Dhcp Disabled
} catch {
    Write-Host "An error occurred while disabling DHCP on $ServerInterfaceName."
    Write-Host "Error: $_"
}

# Get the current IP address of the specified interface

try {
    $currentIP = (Get-NetIPAddress -InterfaceAlias $ServerInterfaceName -AddressFamily IPv4 -ErrorAction Stop).IPAddress
} catch {
    Write-Host "An error occurred while getting the current IP address of $ServerInterfaceName."
    Write-Host "Error: $_"
    $currentIP = $null
}

# If the current IP address is not the same as $ServerIPAddress, update the IP address
try {
    if ($currentIP -ne $ServerIPAddress) {
        Write-Host "Assigning IP address $ServerIPAddress to $ServerInterfaceName with subnet mask of /$ServerSubnetMask."
        New-NetIPAddress -InterfaceAlias $ServerInterfaceName -IPAddress $ServerIPAddress -AddressFamily IPv4 -PrefixLength $ServerSubnetMask -DefaultGateway $DefaultGateway | Out-Null 
    }
    else {
        Write-Host -ForegroundColor Yellow "WARNING: The current IP address of $ServerInterfaceName is already $ServerIPAddress. Skipping IP address assignment."
    }
} catch {
    Write-Host -ForegroundColor Red "ERROR: An error occurred while assigning the IP address to $ServerInterfaceName."
    Write-Host -ForegroundColor Red "$_"
    Exit 1
}

# Set the DNS servers
try{
    Write-Host "Setting the DNS servers to $DnsServer1 and $DnsServer2."
    Set-DnsClientServerAddress -InterfaceAlias $ServerInterfaceName -ServerAddresses ("$DnsServer1","$DnsServer2")
} catch {
    Write-Host -ForegroundColor Red "ERROR: An error occurred while setting the DNS servers."
    Write-Host -ForegroundColor Red "$_"
    Exit 1
}

# Set the power options to never sleep
Try {
    Write-Host "Setting the standby and hibernate power options to never sleep for AC and DC."
    powercfg /change standby-timeout-ac 0
    powercfg /change standby-timeout-dc 0
    powercfg /change hibernate-timeout-ac 0
    powercfg /change hibernate-timeout-dc 0
} catch {
    Write-Host -ForegroundColor Red "ERROR: An error occurred while setting the power options."
    Write-Host -ForegroundColor Red "$_"
    Exit 1
}

# Configure the $ServerUserName account
if (Get-LocalUser -Name $ServerUserName -ErrorAction SilentlyContinue) { 
    Write-Host -ForegroundColor Red "ERROR: Something went wrong. The $ServerUserName account already exists."
    Exit 1
}
else {
    try {
        Write-Host "Creating the $ServerUserName account."
        New-LocalUser -Name $ServerUserName -Password $ServerUserPassword -FullName $ServerUserName -Description "Temporary account used for AutoKape Collections" -ErrorAction Stop | Out-Null
    } catch {
        Write-Host -ForegroundColor Red "ERROR: An error occurred while creating the $ServerUserName account."
        Write-Host -ForegroundColor Red "$_"
        Exit 1
    }
}

# Check if the share already exists
if ( $null -ne (Get-SmbShare -Name $KapeExeSharedFolderName -ErrorAction SilentlyContinue)) {
    Write-Host "The share $KapeExeSharedFolderName already exists."
    Write-Host "Should we delete it and recreate it? (Y/N)"
    while ($true) {
        $response = Read-Host
        if ($response -imatch "^(y|yes)$") {
            break
        }
        elseif ($response -imatch "^(N|No)$") {
            Write-Host "Exiting..."
            exit
        }
        else {
            Write-Host "Invalid response. Please enter Y or N."
        }
    }
    Write-Host "Deleting the existing share."
    try {
        Remove-SmbShare -Name $KapeExeSharedFolderName -Force -ErrorAction Stop
    } catch {
        Write-Host -ForegroundColor Red "ERROR: An error occurred while deleting the existing share."
        Write-Host -ForegroundColor Red "$_"
        Exit 1
    }
}

# Share the KapeExeShare folder
try {
    Write-Host "Sharing the KapeExeShare folder with read access for $ServerUserName"
    New-SmbShare -Name $KapeExeSharedFolderName -Path $KapeExePath -FullAccess "$env:COMPUTERNAME\$ServerUserName" -ErrorAction Stop | Out-Null
} catch {
    Write-Host -ForegroundColor Red "ERROR: An error occurred while sharing the KapeExeShare folder."
    Write-Host -ForegroundColor Red "$_"
    Exit 1
}

# Set NTFS permissions for the KapeExeShare folder
try {
    Write-Host "Setting NTFS permissions for the KapeExeShare folder for $ServerUserName"
    $acl = Get-Acl -Path $KapeExePath
    $accessRule = New-Object System.Security.AccessControl.FileSystemAccessRule($ServerUserName, "Modify", "ContainerInherit, ObjectInherit", "None", "Allow")
    $acl.SetAccessRule($accessRule)
    Set-Acl -Path $KapeExePath -AclObject $acl
} catch {
    Write-Host -ForegroundColor Red "ERROR: An error occurred while setting NTFS permissions for the KapeExeShare folder."
    Write-Host -ForegroundColor Red "$_"
    Exit 1
}

# Check if the share already exists
if ($null -ne (Get-SmbShare -Name $DestinationSharedFolderName -ErrorAction SilentlyContinue)) {
    Write-Host "The share $DestinationSharedFolderName already exists."
    Write-Host "Should we delete it and recreate it? (Y/N)"
    while ($true) {
        $response = Read-Host
        if ($response -imatch "^(y|yes)$") {
            break
        }
        elseif ($response -imatch "^(N|No)$") {
            Write-Host "Exiting..."
            exit
        }
        else {
            Write-Host "Invalid response. Please enter Y or N."
        }
    }
    Write-Host "Deleting the existing share."
    try {
        Remove-SmbShare -Name $DestinationSharedFolderName -Force -ErrorAction Stop
    } catch {
        Write-Host -ForegroundColor Red "ERROR: An error occurred while deleting the existing share."
        Write-Host -ForegroundColor Red "$_"
        Exit 1
    }
}

# Share the DestinationShare folder
try {
    Write-Host "Sharing the DestinationShare folder with write access for $ServerUserName"
    New-SmbShare -Name $DestinationSharedFolderName -Path $DestinationPath -FullAccess $env:COMPUTERNAME\$ServerUserName -ErrorAction Stop | Out-Null
} catch {
    Write-Host -ForegroundColor Red "ERROR: An error occurred while sharing the DestinationShare folder."
    Write-Host -ForegroundColor Red "$_"
    Exit 1
}

# Set NTFS permissions for the DestinationShare folder
try {
    Write-Host "Setting NTFS permissions for the DestinationShare folder for $ServerUserName"
    $acl = Get-Acl -Path $DestinationPath
    $accessRule = New-Object System.Security.AccessControl.FileSystemAccessRule($ServerUserName, "Modify", "ContainerInherit, ObjectInherit", "None", "Allow")
    $acl.SetAccessRule($accessRule)
    Set-Acl -Path $DestinationPath -AclObject $acl
} catch {
    Write-Host -ForegroundColor Red "ERROR: An error occurred while setting NTFS permissions for the DestinationShare folder."
    Write-Host -ForegroundColor Red "$_"
    Exit 1
}

# Open port 445 for all the computers in the TargetComputers array
Write-Host "Opening port 445 for inbound connections from the Target Computers..."
$errorOccurred = $false
foreach ($computer in $TargetComputers) {
    try {
        $ComputerIP = Resolve-DnsName -Name $computer -Type A -ErrorAction Stop | Select-Object -ExpandProperty IPAddress
    }
    catch {
        Write-Host -ForegroundColor Red "ERROR: An error occurred while resolving the IP address for $computer."
        Write-Host -ForegroundColor Red "$_"
        $errorOccurred = $true
        break
    }
    Write-Host "Opening port 445 for traffic from $computer using IP address $ComputerIP."
    try {
        New-NetFirewallRule -DisplayName "KAPE: Allow inbound TCP port 445 from $computer" -Direction Inbound -LocalPort 445 -Protocol TCP -Action Allow -Profile Any -RemoteAddress $computerIP | Out-Null
    }
    catch {
        Write-Host -ForegroundColor Red "ERROR: An error occurred while opening port 445 for $computer."
        Write-Host -ForegroundColor Red " $_"
        $errorOccurred = $true
        break
    }
}

if ($errorOccurred) {
    Write-Host "Script stopping due to errors."
    exit 1
}

Write-Host "Configuration complete. Make sure that all Kape files are in $KapeExeShare and that $DestinationShare is empty."
Write-Host "You can use the transcript file at $ServerTranscript to see the changes that were made.`n"
Write-Host "Please use Run-AutoKape.ps1 from another computer to initiate the collection process." 
Write-Host "The computer where Run-AutoKape.ps1 is run must be able to access $KapeExeShare and $DestinationShare."
Write-Host "It must also be able to access the target computers using PowerShell Remoting."
Write-Host "It will also need to have an administrative credential entered during the script execution, so only use a trusted computer."
Write-Host "Ideally, a dedicated administrative workstation should be used for this purpose."
Write-Host "After collection, you can use Restore-AutoKapeServer.ps1 to help remove the shares and associated permission changes made by this script."


# Stop the transcript
Write-Host "Script stopping at $(Get-Date -Format 'yyyy-MMM-dd HH:mm:ss')."
Stop-Transcript
Write-Host "Transcript completed. Closing transcript file."
