<#
        .SYNOPSIS
        Analysis Windows event logs found in disk images mounted with Mount-TriageVhdx.ps1 with Hayabusa.

        .DESCRIPTION
        Analysis Windows event logs found in disk images mounted with Mount-TriageVhdx.ps1 with Hayabusa. Outputs one CSV per host with all detections found and another for high and critical detections. Provides a combined CSV from all hosts with high and critical detections. 
        
        .PARAMETER MountDirectory
        The folder where your triage disk images are mounted. Each image should be mounted to a subfolder name. This is handled automatically by Mount-TriageVhdx.ps1.

        .PARAMETER ZimmermanToolsDirectory
        Specifies the directory containing the Zimmerman Tools. This folder will often be named 'net6' and contains multiple executables and folders. This script will recursively search the provided folder for executables matching the expected names.

        .PARAMETER OutDirectory
        The folder where your output CSV files will be stored. The folder must already exist. 

        .PARAMETER RepairSrum
        An optional parameter that specifies whether to attempt to repair a copy of the srudb.dat file if the original is corrupted. This is done by copying the file to a temporary location, repairing it, then processing the copy. The default is $false.
        
        .EXAMPLE
        PS> Run-ZTSrumECmd.ps1 -MountDirectory "D:\MountedImagesFolder\" -ZimmermanToolsDirectory "D:\ZimmermanTools" -OutDirectory "F:\OutDir"
        This will run the script with the only required parameters, and default to not attempting to repair the srudb.dat file.

        .EXAMPLE
        PS> Run-ZTSrumECmd.ps1 -MountDirectory "D:\MountedImagesFolder\" -ZimmermanToolsDirectory "D:\ZimmermanTools" -OutDirectory "F:\OutDir" -RepairSrum
        This will run the script with the required parameters and attempt to repair the srudb.dat file if it is corrupted.

        .NOTES
        Written by Steve Anson. This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version. This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. The GNU General Public License can be found at <https://www.gnu.org/licenses/>.

        .LINK
        You can download a script that will download all of Eric Zimmerman's tools from here: https://github.com/EricZimmerman/Get-ZimmermanTools
#>
        


    param (
        [Parameter(Position=0,mandatory=$true)]
        $MountDirectory,
        [Parameter(Position=1,mandatory=$true)]
        $ZimmermanToolsDirectory,
        [Parameter(Position=2,mandatory=$true)]
        $OutDirectory,
        [Parameter(Position=3,mandatory=$false)]
        [switch]$RepairSrum
    )
    
    # Check if the required directories exist
    if (-not (Test-Path $MountDirectory)) {
        Write-Host "Mount directory '$MountDirectory' does not exist. Exiting..."
        exit 1
    }
    
    if (-not (Test-Path $ZimmermanToolsDirectory)) {
        Write-Host "Zimmerman Tools directory '$ZimmermanToolsDirectory' does not exist. Exiting..."
        exit 1
    }
    
    # Declare script variables
    $script:processes = @()
    $script:processes2 = @()


    # Recursively search through $ZimmermanToolsDirectory to find SRUMECmd.exe

    $SrumECmdExe = Get-ChildItem -Path $ZimmermanToolsDirectory -Recurse -Filter "SrumECmd.exe" | Select-Object -ExpandProperty FullName

    if (-not $SrumECmdExe) {
        Write-Host "The SrumECmd.exe executable is missing"
        exit 1
    }

    # Declare Functions

    function RunSrumECmdFirst {
        param (
            $HostDir
        )

        # Declare variables
        $SrumECmdExe = Get-ChildItem -Path $ZimmermanToolsDirectory -Recurse -Filter "SrumECmd.exe" | Select-Object -ExpandProperty FullName
        $SoftwareRegistryHivePath = Join-Path -Path $MountDirectory -ChildPath "$HostDir\C\Windows\System32\config\SOFTWARE"
        $SrumOutputTmpFile = Join-Path -Path $OutDirectory -ChildPath "$($HostDir)\srum.tmp"
        $SrumPath = Join-Path -Path $MountDirectory -ChildPath "$HostDir\C\Windows\System32\sru\srudb.dat"
        
        If (-not (Test-Path $OutDirectory$HostDir)) {
            [void](New-Item -Path $OutDirectory -Name $HostDir -ItemType Directory)
        }

        # Check for SRUM data and process it
        if (Test-Path $SrumPath) {
            Write-Host "Processing SRUM data for $HostDir"
            # Start a new process for each host, storing the output from $SrumECmdExe in a temporary file for error checking
            $process = Start-Process -RedirectStandardOutput $SrumOutputTmpFile -PassThru -FilePath $SrumECmdExe -ArgumentList "-f `"$SrumPath`" -r `"$SoftwareRegistryHivePath`" --csv `"$($OutDirectory)$($HostDir)\srum`"" 
            $script:processes += $process
            
        }
        else {
            Write-Host -ForegroundColor Yellow "WARNING: No SRUM data found for $HostDir"
        }
    }

    function RunSrumECmdSecond {
        param (
            $HostDir
        )

        #Declare variables
        $SrumECmdExe = Get-ChildItem -Path $ZimmermanToolsDirectory -Recurse -Filter "SrumECmd.exe" | Select-Object -ExpandProperty FullName
        $SoftwareRegistryHivePath = Join-Path -Path $MountDirectory -ChildPath "$HostDir\C\Windows\System32\config\SOFTWARE"
        $SrumOutputTmpFile = Join-Path -Path $OutDirectory -ChildPath "$($HostDir)\srum.tmp"
        $SrumPath = Join-Path -Path $MountDirectory -ChildPath "$HostDir\C\Windows\System32\sru\srudb.dat"


        # Srudb.dat is often corrupted. This will attempt to repair it if the -RepairSrum switch is used
        # Check for error in SRUM processing
        If (Get-Content $SrumOutputTmpFile | Select-String -Pattern "Error processing file!") {
            Write-Host -ForegroundColor Red "ERROR: Processing SRUM data for $HostDir failed. This is likely due to the srudb.dat file being corrupted."
            
            # Attempt to repair the file if the -RepairSrum switch is used
            If ($RepairSrum) {
                Write-Host -ForegroundColor Yellow "Attempting to repair a copy of srudb.dat from $HostDir..."
                Remove-Item $SrumOutputTmpFile #remove old .tmp file since a new one will be created
                $SrumPathTmp = Join-Path -Path $OutDirectory -ChildPath "$($HostDir)\srum_repair\srudb.dat"
                If (-not (Test-Path $OutDirectory$HostDir\srum_repair)) {
                    [void](New-Item -Path $OutDirectory$HostDir -Name "srum_repair" -ItemType Directory)
                }
                Copy-Item -Path $SrumPath -Destination $SrumPathTmp
                Start-Process esentutl.exe -ArgumentList "-r `"$($SrumPathTmp) /i`"" -Wait
                Start-Process esentutl.exe -ArgumentList "-p `"$($SrumPathTmp)`"" -Wait
                Write-Host -ForegroundColor Yellow "Repair attempt complete. Reprocessing SRUM data for $HostDir to determine if repair was successful.`nAny additional errors will be reported in this window when reprocessing completes."
                $process = Start-Process -RedirectStandardOutput $SrumOutputTmpFile -PassThru -FilePath $SrumECmdExe -ArgumentList "-f `"$SrumPathTmp`" -r `"$SoftwareRegistryHivePath`" --csv `"$($OutDirectory)$($HostDir)\srum_results`"" 
                $script:processes2 += $process
            }
            else {
                Write-Host -ForegroundColor Red "You can find additional details in $SrumOutputTmpFile"
                Write-Host -ForegroundColor Red "You can attempt to repair the file by running this script with the -RepairSrum switch."
            }
        }
        else {
            try {
                Remove-Item $SrumOutputTmpFile -ErrorAction Stop
            }
            catch {
                Write-Host -ForegroundColor Yellow "WARNING: Unable to remove $SrumOutputTmpFile during cleanup. You may need to manually remove it if desired."
            }
            
        }
    }

    function RunSrumECmdThird {
        param (
            $HostDir
        )

        #Declare variables
        $SrumOutputTmpFile = Join-Path -Path $OutDirectory -ChildPath "$($HostDir)\srum.tmp"

        # Check for error in SRUM processing
        If (Get-Content $SrumOutputTmpFile | Select-String -Pattern "Error processing file!") {
            Write-Host -ForegroundColor Red "ERROR: *****Repairing srudb.dat on $HostDir failed.***** `nYou will need to manually repair the file."
            Write-Host -ForegroundColor Red "You can find additional details in $SrumOutputTmpFile"
        }
        else {
            try {
                Remove-Item $SrumOutputTmpFile -ErrorAction Stop
            }
            catch {
                Write-Host -ForegroundColor Yellow "WARNING: Unable to remove $SrumOutputTmpFile during cleanup. You may need to manually remove it if desired."
            }
        }
    }
    
    # Main program

    #Iterate through all mounted image directories and process with SrumECmd
    Write-Host "Now processing all images mounted in $MountDirectory"
    $HostDir = Get-ChildItem -Path "$MountDirectory" -Directory -Name
    $HostDir | ForEach-Object {
    RunSrumECmdFirst -HostDir $_
    }

    Write-Host "Each new window is processing a task. `nAs each task completes, the associated window will close."
    Write-Host "Until then, the data is still being processed. Please wait..."

    # Monitor processes and update progress bar
    while ($script:processes | Where-Object { $_.HasExited -eq $false }) {
        Start-Sleep -Seconds 1 # Wait before checking again

        # Calculate completed processes
        $completed = ($script:processes | Where-Object { $_.HasExited }).Count 
        $total = $script:processes.Count 
        $percentComplete = ($completed / $total) * 100
    
        Write-Progress -ID 1 -Activity "Intitial Processing" -Status "Waiting for initial processing to complete" -PercentComplete $percentComplete
    }
    
    Write-Progress -ID 2 -Activity "Additional Processing and Clean Up" -Status "Checking results and repairing if -RepairSrum was specified." -PercentComplete 0

    # Iterate through all mounted image directories and process data with SrumECmd
    # Since srudb.dat is often corrupted, this will attempt to repair it if the -RepairSrum switch is used
    $HostDir = Get-ChildItem -Path "$MountDirectory" -Directory -Name
    $HostDir | ForEach-Object {
        RunSrumECmdSecond -HostDir $_
     }


    # Monitor processes and update progress bar
    while ($script:Processes2 | Where-Object { $_.HasExited -eq $false }) {
        Start-Sleep -Seconds 1 # Wait before checking again

        # Calculate completed processes
        $completed = ($script:Processes2 | Where-Object { $_.HasExited }).Count
        $total = $script:Processes2.Count
        $percentComplete = ($completed / $total) * 100

        Write-Progress -ID 2 -Activity "Additional Processing and Clean Up" -Status "Checking results and repairing if -RepairSrum was specified." -PercentComplete $percentComplete
    }

    # Final cleanup if repair was needed
    $HostDir = Get-ChildItem -Path "$MountDirectory" -Directory -Name
    $HostDir | ForEach-Object {
        if ($RepairSrum) { 
            RunSrumECmdThird -HostDir $_
        }
     }

    # Report completion
    Write-Host "Processing is completed.`nThe results are stored in $($OutDirectory) with a subfolder for each system."
    if ($RepairSrum) {Write-Host "Details related to the repair attempt are stored in a subfolder named srum_repair."}
    Write-Host "Results of the processing are stored in a subfolder named srum_results.`nYou may now begin your analysis."